/* SPDX-License-Identifier: BSD-3-Clause
 * Copyright(c) 2001-2018
 */

#ifndef _TXGBE_TYPE_H_
#define _TXGBE_TYPE_H_

/*
 * Driver Configuration
 */
/* DCB configuration defines */
#define TXGBE_DCB_TC_MAX        TXGBE_MAX_UP
#define TXGBE_DCB_UP_MAX        TXGBE_MAX_UP
#define TXGBE_DCB_BWG_MAX       TXGBE_MAX_UP
#define TXGBE_DCB_BW_PERCENT    100

#define TXGBE_LINK_UP_TIME         90 /* 9.0 Seconds */
#define TXGBE_AUTO_NEG_TIME		45 /* 4.5 Seconds */

#define TXGBE_RX_HDR_SIZE    256
#define TXGBE_RX_BUF_SIZE    2048

#define TXGBE_FRAME_SIZE_MAX       (9728) /* Maximum frame size, +FCS */
#define TXGBE_FRAME_SIZE_DFT       (1518) /* Default frame size, +FCS */
#define TXGBE_NUM_POOL             (64)
#define TXGBE_PBRXSIZE_MAX         0x00080000 /* 512KB Packet Buffer */
#define TXGBE_TXPKTSIZE_MAX        (10)
#define TXGBE_PBTXSIZE_MAX         0x00028000 /* 160KB Packet Buffer */
#define TXGBE_FDIR_DROP_QUEUE      127
#define TXGBE_MAX_FTQF_FILTERS     128
#define TXGBE_TXPKT_SIZE_MAX    0xA /* Max Tx Packet size */
#define TXGBE_MAX_UP            8
#define TXGBE_MAX_QP (128)

#define TXGBE_MAX_UTA                   128

#define TXGBE_FDIR_INIT_DONE_POLL		10
#define TXGBE_FDIRCMD_CMD_POLL			10
#define TXGBE_MD_TIMEOUT 1000
#define TXGBE_SPI_TIMEOUT  1000
#define TXGBE_VF_INIT_TIMEOUT	200 /* Number of retries to clear RSTI */
#define TXGBE_PCI_MASTER_DISABLE_TIMEOUT	800

#define TXGBE_MAX_MSIX_VECTORS_RAPTOR	0x40

#define TXGBE_ALIGN		128 /* as intel did */

/*
 * The following is a brief description of the error categories used by the
 * ERROR_REPORT* macros.
 *
 * - TXGBE_ERROR_INVALID_STATE
 * This category is for errors which represent a serious failure state that is
 * unexpected, and could be potentially harmful to device operation. It should
 * not be used for errors relating to issues that can be worked around or
 * ignored.
 *
 * - TXGBE_ERROR_POLLING
 * This category is for errors related to polling/timeout issues and should be
 * used in any case where the timeout occured, or a failure to obtain a lock, or
 * failure to receive data within the time limit.
 *
 * - TXGBE_ERROR_CAUTION
 * This category should be used for reporting issues that may be the cause of
 * other errors, such as temperature warnings. It should indicate an event which
 * could be serious, but hasn't necessarily caused problems yet.
 *
 * - TXGBE_ERROR_SOFTWARE
 * This category is intended for errors due to software state preventing
 * something. The category is not intended for errors due to bad arguments, or
 * due to unsupported features. It should be used when a state occurs which
 * prevents action but is not a serious issue.
 *
 * - TXGBE_ERROR_ARGUMENT
 * This category is for when a bad or invalid argument is passed. It should be
 * used whenever a function is called and error checking has detected the
 * argument is wrong or incorrect.
 *
 * - TXGBE_ERROR_UNSUPPORTED
 * This category is for errors which are due to unsupported circumstances or
 * configuration issues. It should not be used when the issue is due to an
 * invalid argument, but for when something has occurred that is unsupported
 * (Ex: Flow control autonegotiation or an unsupported SFP+ module.)
 */

#include "txgbe_status.h"
#include "txgbe_osdep.h"
#include "txgbe_devids.h"

/* Sensors for AMLITE PVT(Process Voltage Temperature) */
#define TXGBE_AML_INTR_RAW_HI           0x10300
#define TXGBE_AML_INTR_RAW_ME           0x10304
#define TXGBE_AML_INTR_RAW_LO           0x10308
#define TXGBE_AML_TS_CTL1               0x10330
#define TXGBE_AML_TS_CTL2               0x10334
#define TXGBE_AML_TS_ENA                0x10338
#define TXGBE_AML_TS_STS                0x1033C
#define TXGBE_AML_INTR_HIGH_EN          0x10318
#define TXGBE_AML_INTR_MED_EN           0x1031C
#define TXGBE_AML_INTR_LOW_EN           0x10320
#define TXGBE_AML_INTR_HIGH_STS         0x1030C
#define TXGBE_AML_INTR_MED_STS          0x10310
#define TXGBE_AML_INTR_LOW_STS          0x10314

#define TXGBE_AML_TS_STS_VLD            0x1000
#define TXGBE_AML_INTR_EN_HI            0x00000002U
#define TXGBE_AML_INTR_EN_ME            0x00000001U
#define TXGBE_AML_INTR_EN_LO            0x00000001U
#define TXGBE_AML_INTR_CL_HI            0x00000002U
#define TXGBE_AML_INTR_CL_ME            0x00000001U
#define TXGBE_AML_INTR_CL_LO            0x00000001U
#define TXGBE_AML_EVAL_MODE_MASK        0x010U
#define TXGBE_AML_CAL_MODE_MASK         0x08U
#define TXGBE_AML_ALARM_THRE_MASK       0x1FFE0000U
#define TXGBE_AML_DALARM_THRE_MASK      0x0001FFE0U

/* Override this by setting IOMEM in your txgbe_osdep.h header */

#define CL74_KRTR_TRAINNING_TIMEOUT	3000	/* 3000ms c74 trainning timeout */
#define AN_TRAINNING_MODE		0	/* 0: not dis an 1: dis an */

struct txgbe_thermal_diode_data {
	//u8 location;
	s16 temp;
	//u8 caution_thresh;
	//u8 max_op_thresh;
	s16 alarm_thresh;
	s16 dalarm_thresh;
};

struct txgbe_thermal_sensor_data {
	struct txgbe_thermal_diode_data sensor[1];
};

struct txgbe_nvm_version {
	u32 etk_id;
	u8  nvm_major;
	u16 nvm_minor;
	u8  nvm_id;

	bool oem_valid;
	u8   oem_major;
	u8   oem_minor;
	u16  oem_release;

	bool or_valid;
	u8  or_major;
	u16 or_build;
	u8  or_patch;
};

/* Power Management */
/* DMA Coalescing configuration */
struct txgbe_dmac_config {
	u16	watchdog_timer; /* usec units */
	bool	fcoe_en;
	u32	link_speed;
	u8	fcoe_tc;
	u8	num_tcs;
};

/* Packet buffer allocation strategies */
enum {
	PBA_STRATEGY_EQUAL	= 0, /* Distribute PB space equally */
#define PBA_STRATEGY_EQUAL	PBA_STRATEGY_EQUAL
	PBA_STRATEGY_WEIGHTED	= 1, /* Weight front half of TCs */
#define PBA_STRATEGY_WEIGHTED	PBA_STRATEGY_WEIGHTED
};

/* Physical layer type */
#define TXGBE_PHYSICAL_LAYER_UNKNOWN            0
#define TXGBE_PHYSICAL_LAYER_10GBASE_T          0x00001
#define TXGBE_PHYSICAL_LAYER_1000BASE_T         0x00002
#define TXGBE_PHYSICAL_LAYER_100BASE_TX         0x00004
#define TXGBE_PHYSICAL_LAYER_SFP_PLUS_CU        0x00008
#define TXGBE_PHYSICAL_LAYER_10GBASE_LR         0x00010
#define TXGBE_PHYSICAL_LAYER_10GBASE_LRM        0x00020
#define TXGBE_PHYSICAL_LAYER_10GBASE_SR         0x00040
#define TXGBE_PHYSICAL_LAYER_10GBASE_KX4        0x00080
#define TXGBE_PHYSICAL_LAYER_10GBASE_CX4        0x00100
#define TXGBE_PHYSICAL_LAYER_1000BASE_KX        0x00200
#define TXGBE_PHYSICAL_LAYER_1000BASE_BX        0x00400
#define TXGBE_PHYSICAL_LAYER_10GBASE_KR         0x00800
#define TXGBE_PHYSICAL_LAYER_10GBASE_XAUI       0x01000
#define TXGBE_PHYSICAL_LAYER_SFP_ACTIVE_DA      0x02000
#define TXGBE_PHYSICAL_LAYER_1000BASE_SX        0x04000
#define TXGBE_PHYSICAL_LAYER_10BASE_T           0x08000
#define TXGBE_PHYSICAL_LAYER_2500BASE_KX        0x10000

/* Software ATR hash keys */
#define TXGBE_ATR_BUCKET_HASH_KEY       0x3DAD14E2
#define TXGBE_ATR_SIGNATURE_HASH_KEY    0x174D3614

/* Software ATR input stream values and masks */
#define TXGBE_ATR_HASH_MASK             0x7fff
#define TXGBE_ATR_L3TYPE_MASK           0x4
#define TXGBE_ATR_L3TYPE_IPV4           0x0
#define TXGBE_ATR_L3TYPE_IPV6           0x4
#define TXGBE_ATR_L4TYPE_MASK           0x3
#define TXGBE_ATR_L4TYPE_UDP            0x1
#define TXGBE_ATR_L4TYPE_TCP            0x2
#define TXGBE_ATR_L4TYPE_SCTP           0x3
enum txgbe_atr_flow_type {
	TXGBE_ATR_FLOW_TYPE_IPV4 = 0x0,
	TXGBE_ATR_FLOW_TYPE_UDPV4 = 0x1,
	TXGBE_ATR_FLOW_TYPE_TCPV4 = 0x2,
	TXGBE_ATR_FLOW_TYPE_SCTPV4 = 0x3,
	TXGBE_ATR_FLOW_TYPE_IPV6 = 0x4,
	TXGBE_ATR_FLOW_TYPE_UDPV6 = 0x5,
	TXGBE_ATR_FLOW_TYPE_TCPV6 = 0x6,
	TXGBE_ATR_FLOW_TYPE_SCTPV6 = 0x7,
};
#define TXGBE_ATR_TYPE_MASK_TUN			0x80
#define TXGBE_ATR_TYPE_MASK_TUN_OUTIP		0x40
#define TXGBE_ATR_TYPE_MASK_TUN_TYPE		0x20
#define TXGBE_ATR_TYPE_MASK_L3P			0x10
#define TXGBE_ATR_TYPE_MASK_L4P			0x08

/* Flow Director ATR input struct. */
struct txgbe_atr_input {
	/*
	 * Byte layout in order, all values with MSB first:
	 *
	 * vm_pool      - 1 byte
	 * flow_type    - 1 byte
	 * pkt_type	- 2 bytes
	 * src_ip       - 16 bytes
	 * dst_ip       - 16 bytes
	 * src_port     - 2 bytes
	 * dst_port     - 2 bytes
	 * flex_bytes   - 2 bytes
	 * bkt_hash     - 2 bytes
	 */
	u8 vm_pool;
	u8 flow_type;
	__be16 pkt_type;
	__be32 dst_ip[4];
	__be32 src_ip[4];
	__be16 src_port;
	__be16 dst_port;
	__be16 flex_bytes;
	__be16 bkt_hash;
};

/*
 * Unavailable: The FCoE Boot Option ROM is not present in the flash.
 * Disabled: Present; boot order is not set for any targets on the port.
 * Enabled: Present; boot order is set for at least one target on the port.
 */
enum txgbe_fcoe_boot_status {
	txgbe_fcoe_bootstatus_disabled = 0,
	txgbe_fcoe_bootstatus_enabled = 1,
	txgbe_fcoe_bootstatus_unavailable = 0xFFFF
};

enum txgbe_eeprom_type {
	txgbe_eeprom_unknown = 0,
	txgbe_eeprom_spi,
	txgbe_eeprom_flash,
	txgbe_eeprom_none /* No NVM support */
};

enum txgbe_mac_type {
	txgbe_mac_unknown = 0,
	txgbe_mac_raptor,
	txgbe_mac_aml,
	txgbe_mac_aml40,
	txgbe_mac_raptor_vf,
	txgbe_mac_aml_vf,
	txgbe_num_macs
};

enum txgbe_phy_type {
	txgbe_phy_unknown = 0,
	txgbe_phy_none,
	txgbe_phy_tn,
	txgbe_phy_aq,
	txgbe_phy_ext_1g_t,
	txgbe_phy_cu_mtd,
	txgbe_phy_cu_unknown,
	txgbe_phy_qt,
	txgbe_phy_xaui,
	txgbe_phy_nl,
	txgbe_phy_sfp_tyco_passive,
	txgbe_phy_sfp_unknown_passive,
	txgbe_phy_sfp_unknown_active,
	txgbe_phy_sfp_avago,
	txgbe_phy_sfp_ftl,
	txgbe_phy_sfp_ftl_active,
	txgbe_phy_sfp_unknown,
	txgbe_phy_sfp_intel,
	txgbe_phy_qsfp_unknown_passive,
	txgbe_phy_qsfp_unknown_active,
	txgbe_phy_qsfp_intel,
	txgbe_phy_qsfp_unknown,
	txgbe_phy_sfp_unsupported, /*Enforce bit set with unsupported module*/
	txgbe_phy_sgmii,
	txgbe_phy_fw,
	txgbe_phy_generic
};

/*
 * SFP+ module type IDs:
 *
 * ID	Module Type
 * =============
 * 0    SFP_DA_CU
 * 1    SFP_SR
 * 2    SFP_LR
 * 3    SFP_DA_CU_CORE0 - chip-specific
 * 4    SFP_DA_CU_CORE1 - chip-specific
 * 5    SFP_SR/LR_CORE0 - chip-specific
 * 6    SFP_SR/LR_CORE1 - chip-specific
 */
enum txgbe_sfp_type {
	txgbe_sfp_type_unknown = 0,
	txgbe_sfp_type_da_cu = 1,
	txgbe_sfp_type_sr = 2,
	txgbe_sfp_type_lr = 3,
	txgbe_sfp_type_da_cu_core0 = 4,
	txgbe_sfp_type_da_cu_core1 = 5,
	txgbe_sfp_type_srlr_core0 = 6,
	txgbe_sfp_type_srlr_core1 = 7,
	txgbe_sfp_type_da_act_lmt_core0 = 8,
	txgbe_sfp_type_da_act_lmt_core1 = 9,
	txgbe_sfp_type_1g_cu_core0 = 10,
	txgbe_sfp_type_1g_cu_core1 = 11,
	txgbe_sfp_type_1g_sx_core0 = 12,
	txgbe_sfp_type_1g_sx_core1 = 13,
	txgbe_sfp_type_1g_lx_core0 = 14,
	txgbe_sfp_type_1g_lx_core1 = 15,
	txgbe_sfp_type_25g_sr_core0 = 16,
	txgbe_sfp_type_25g_sr_core1 = 17,
	txgbe_sfp_type_25g_lr_core0 = 18,
	txgbe_sfp_type_25g_lr_core1 = 19,
	txgbe_sfp_type_25g_aoc_core0 = 20,
	txgbe_sfp_type_25g_aoc_core1 = 21,
	txgbe_qsfp_type_40g_cu_core0 = 22,
	txgbe_qsfp_type_40g_cu_core1 = 23,
	txgbe_qsfp_type_40g_sr_core0 = 24,
	txgbe_qsfp_type_40g_sr_core1 = 25,
	txgbe_qsfp_type_40g_lr_core0 = 26,
	txgbe_qsfp_type_40g_lr_core1 = 27,
	txgbe_sfp_type_not_present = 0xFFFE,
	txgbe_sfp_type_not_known = 0xFFFF
};

enum txgbe_media_type {
	txgbe_media_type_unknown = 0,
	txgbe_media_type_fiber,
	txgbe_media_type_fiber_qsfp,
	txgbe_media_type_copper,
	txgbe_media_type_backplane,
	txgbe_media_type_cx4,
	txgbe_media_type_virtual
};

/* Flow Control Settings */
enum txgbe_fc_mode {
	txgbe_fc_none = 0,
	txgbe_fc_rx_pause,
	txgbe_fc_tx_pause,
	txgbe_fc_full,
	txgbe_fc_default
};

/* Smart Speed Settings */
#define TXGBE_SMARTSPEED_MAX_RETRIES	3
enum txgbe_smart_speed {
	txgbe_smart_speed_auto = 0,
	txgbe_smart_speed_on,
	txgbe_smart_speed_off
};

/* PCI bus types */
enum txgbe_bus_type {
	txgbe_bus_type_unknown = 0,
	txgbe_bus_type_pci,
	txgbe_bus_type_pcix,
	txgbe_bus_type_pci_express,
	txgbe_bus_type_internal,
	txgbe_bus_type_reserved
};

/* PCI bus speeds */
enum txgbe_bus_speed {
	txgbe_bus_speed_unknown	= 0,
	txgbe_bus_speed_33	= 33,
	txgbe_bus_speed_66	= 66,
	txgbe_bus_speed_100	= 100,
	txgbe_bus_speed_120	= 120,
	txgbe_bus_speed_133	= 133,
	txgbe_bus_speed_2500	= 2500,
	txgbe_bus_speed_5000	= 5000,
	txgbe_bus_speed_8000	= 8000,
	txgbe_bus_speed_reserved
};

/* PCI bus widths */
enum txgbe_bus_width {
	txgbe_bus_width_unknown	= 0,
	txgbe_bus_width_pcie_x1	= 1,
	txgbe_bus_width_pcie_x2	= 2,
	txgbe_bus_width_pcie_x4	= 4,
	txgbe_bus_width_pcie_x8	= 8,
	txgbe_bus_width_32	= 32,
	txgbe_bus_width_64	= 64,
	txgbe_bus_width_reserved
};

struct txgbe_hw;

struct txgbe_addr_filter_info {
	u32 num_mc_addrs;
	u32 rar_used_count;
	u32 mta_in_use;
	u32 overflow_promisc;
	bool user_set_promisc;
};

/* Bus parameters */
struct txgbe_bus_info {
	s32 (*get_bus_info)(struct txgbe_hw *);
	void (*set_lan_id)(struct txgbe_hw *);

	enum txgbe_bus_speed speed;
	enum txgbe_bus_width width;
	enum txgbe_bus_type type;

	u16 func;
	u8 lan_id;
	u16 instance_id;
};

/* Flow control parameters */
struct txgbe_fc_info {
	u32 high_water[TXGBE_DCB_TC_MAX]; /* Flow Ctrl High-water */
	u32 low_water[TXGBE_DCB_TC_MAX]; /* Flow Ctrl Low-water */
	u16 pause_time; /* Flow Control Pause timer */
	u8 mac_ctrl_frame_fwd; /* Forward MAC control frames */
	bool send_xon; /* Flow control send XON */
	bool strict_ieee; /* Strict IEEE mode */
	bool disable_fc_autoneg; /* Do not autonegotiate FC */
	bool fc_was_autonegged; /* Is current_mode the result of autonegging? */
	enum txgbe_fc_mode current_mode; /* FC mode in effect */
	enum txgbe_fc_mode requested_mode; /* FC mode requested by caller */
};

/* Statistics counters collected by the MAC */
/* PB[] RxTx */
struct txgbe_pb_stats {
	u64 tx_pb_xon_packets; //pxontxc;
	u64 rx_pb_xon_packets; //pxonrxc;
	u64 tx_pb_xoff_packets; //pxofftxc;
	u64 rx_pb_xoff_packets; //pxoffrxc;
	u64 rx_pb_dropped; //mpc;
	u64 rx_pb_mbuf_alloc_errors; //rnbc;
	u64 tx_pb_xon2off_packets; //pxon2offc;
};

/* QP[] RxTx */
struct txgbe_qp_stats {
	u64 rx_qp_packets; //qprc;
	u64 tx_qp_packets; //qptc;
	u64 rx_qp_bytes; //qbrc;
	u64 tx_qp_bytes; //qbtc;
	u64 rx_qp_mc_packets; //qprdc;
};

struct txgbe_hw_stats {
	/* MNG RxTx */
	u64 mng_bmc2host_packets; //b2ospc
	u64 mng_host2bmc_packets; //o2bspc
	/* Basix RxTx */
	u64 rx_packets; //gprc;
	u64 tx_packets; //gptc;
	u64 rx_bytes; //gorc;
	u64 tx_bytes; //gotc;
	u64 rx_total_bytes; //tor;
	u64 rx_total_packets; //tpr;
	u64 tx_total_packets; //tpt;
	u64 rx_total_missed_packets; //mpctotal;
	u64 rx_broadcast_packets; //bprc;
	u64 tx_broadcast_packets; //bptc;
	u64 rx_multicast_packets; //mprc;
	u64 tx_multicast_packets; //mptc;
	u64 rx_management_packets; //b2ogprc;
	u64 tx_management_packets; //o2bgptc;
	u64 rx_management_dropped; //mngpdc;
	u64 rx_dma_drop;

	/* Basic Error */
	u64 rx_rdb_drop;
	u64 rx_crc_errors; //crcerrs;
	u64 rx_illegal_byte_errors; //illerrc;
	u64 rx_error_bytes; //errbc;
	u64 rx_mac_short_packet_dropped; //mspdc;
	u64 rx_length_errors; //rlec;
	u64 rx_undersize_errors; //ruc;
	u64 rx_fragment_errors; //rfc;
	u64 rx_oversize_cnt;
	u64 rx_jabber_errors; //rjc;
	u64 rx_l3_l4_xsum_error; //xec;
	u64 mac_local_errors; //mlfc;
	u64 mac_remote_errors; //mrfc;

	/* Flow Director */
	u64 flow_director_added_filters; //fdirustat_add;
	u64 flow_director_removed_filters; //fdirustat_remove;
	u64 flow_director_filter_add_errors; //fdirfstat_fadd;
	u64 flow_director_filter_remove_errors; //fdirfstat_fremove;
	u64 flow_director_matched_filters; //fdirmatch;
	u64 flow_director_missed_filters; //fdirmiss;

	/* FCoE */
	u64 rx_fcoe_crc_errors; //fccrc;
	u64 rx_fcoe_mbuf_allocation_errors; //fclast;
	u64 rx_fcoe_dropped; //fcoerpdc;
	u64 rx_fcoe_packets; //fcoeprc;
	u64 tx_fcoe_packets; //fcoeptc;
	u64 rx_fcoe_bytes; //fcoedwrc;
	u64 tx_fcoe_bytes; //fcoedwtc;
	u64 rx_fcoe_no_ddp; //fcoe_noddp;
	u64 rx_fcoe_no_ddp_ext_buff; //fcoe_noddp_ext_buff;

	/* MACSEC */
	u64 tx_macsec_pkts_untagged;
	u64 tx_macsec_pkts_encrypted;
	u64 tx_macsec_pkts_protected;
	u64 tx_macsec_octets_encrypted;
	u64 tx_macsec_octets_protected;
	u64 rx_macsec_pkts_untagged;
	u64 rx_macsec_pkts_badtag;
	u64 rx_macsec_pkts_nosci;
	u64 rx_macsec_pkts_unknownsci;
	u64 rx_macsec_octets_decrypted;
	u64 rx_macsec_octets_validated;
	u64 rx_macsec_sc_pkts_unchecked;
	u64 rx_macsec_sc_pkts_delayed;
	u64 rx_macsec_sc_pkts_late;
	u64 rx_macsec_sa_pkts_ok;
	u64 rx_macsec_sa_pkts_invalid;
	u64 rx_macsec_sa_pkts_notvalid;
	u64 rx_macsec_sa_pkts_unusedsa;
	u64 rx_macsec_sa_pkts_notusingsa;

	/* MAC RxTx */
	u64 rx_size_64_packets; //prc64;
	u64 rx_size_65_to_127_packets; //prc127;
	u64 rx_size_128_to_255_packets; //prc255;
	u64 rx_size_256_to_511_packets; //prc511;
	u64 rx_size_512_to_1023_packets; //prc1023;
	u64 rx_size_1024_to_max_packets; //prc1522;
	u64 tx_size_64_packets; //ptc64;
	u64 tx_size_65_to_127_packets; //ptc127;
	u64 tx_size_128_to_255_packets; //ptc255;
	u64 tx_size_256_to_511_packets; //ptc511;
	u64 tx_size_512_to_1023_packets; //ptc1023;
	u64 tx_size_1024_to_max_packets; //ptc1522;

	/* Flow Control */
	u64 tx_xon_packets; //lxontxc;
	u64 rx_xon_packets; //lxonrxc;
	u64 tx_xoff_packets; //lxofftxc;
	u64 rx_xoff_packets; //lxoffrxc;

	/* PB[] RxTx */
	struct {
		u64 rx_up_packets; //qprc;
		u64 tx_up_packets; //qptc;
		u64 rx_up_bytes; //qbrc;
		u64 tx_up_bytes; //qbtc;
		u64 rx_up_drop_packets; //qprdc;

		u64 tx_up_xon_packets; //pxontxc;
		u64 rx_up_xon_packets; //pxonrxc;
		u64 tx_up_xoff_packets; //pxofftxc;
		u64 rx_up_xoff_packets; //pxoffrxc;
		u64 rx_up_dropped; //mpc;
		u64 rx_up_mbuf_alloc_errors; //rnbc;
		u64 tx_up_xon2off_packets; //pxon2offc;
	} up[TXGBE_MAX_UP];

	/* QP[] RxTx */
	struct {
		u64 rx_qp_packets;
		u64 tx_qp_packets;
		u64 rx_qp_bytes;
		u64 tx_qp_bytes;
		u64 rx_qp_mc_packets;
	} qp[TXGBE_MAX_QP];

};

/* iterator type for walking multicast address lists */
typedef u8* (*txgbe_mc_addr_itr) (struct txgbe_hw *hw, u8 **mc_addr_ptr,
				  u32 *vmdq);

struct txgbe_link_info {
	s32 (*read_link)(struct txgbe_hw *, u8 addr, u16 reg, u16 *val);
	s32 (*read_link_unlocked)(struct txgbe_hw *, u8 addr, u16 reg,
				  u16 *val);
	s32 (*write_link)(struct txgbe_hw *, u8 addr, u16 reg, u16 val);
	s32 (*write_link_unlocked)(struct txgbe_hw *, u8 addr, u16 reg,
				   u16 val);

	u8 addr;
};

struct txgbe_rom_info {
	s32 (*init_params)(struct txgbe_hw *);
	s32 (*read16)(struct txgbe_hw *, u32, u16 *);
	s32 (*readw_sw)(struct txgbe_hw *, u32, u16 *);
	s32 (*readw_buffer)(struct txgbe_hw *, u32, u32, void *);
	s32 (*read32)(struct txgbe_hw *, u32, u32 *);
	s32 (*read_buffer)(struct txgbe_hw *, u32, u32, void *);
	s32 (*write16)(struct txgbe_hw *, u32, u16);
	s32 (*writew_sw)(struct txgbe_hw *, u32, u16);
	s32 (*writew_buffer)(struct txgbe_hw *, u32, u32, void *);
	s32 (*write32)(struct txgbe_hw *, u32, u32);
	s32 (*write_buffer)(struct txgbe_hw *, u32, u32, void *);
	s32 (*validate_checksum)(struct txgbe_hw *, u16 *);
	s32 (*update_checksum)(struct txgbe_hw *);
	s32 (*calc_checksum)(struct txgbe_hw *);

	enum txgbe_eeprom_type type;
	u32 semaphore_delay;
	u16 word_size;
	u16 address_bits;
	u16 word_page_size;
	u16 ctrl_word_3;

	u32 sw_addr; //fixme
};


struct txgbe_flash_info {
	s32 (*init_params)(struct txgbe_hw *);
	s32 (*read_buffer)(struct txgbe_hw *, u32, u32, u32 *);
	s32 (*write_buffer)(struct txgbe_hw *, u32, u32, u32 *);
	u32 semaphore_delay;
	u32 dword_size;
	u16 address_bits;
};

#define TXGBE_FLAGS_DOUBLE_RESET_REQUIRED	0x01
struct txgbe_mac_info {
	s32 (*init_hw)(struct txgbe_hw *);
	s32 (*reset_hw)(struct txgbe_hw *);
	s32 (*start_hw)(struct txgbe_hw *);
	s32 (*stop_hw)(struct txgbe_hw *);
	s32 (*clear_hw_cntrs)(struct txgbe_hw *);
	void (*enable_relaxed_ordering)(struct txgbe_hw *);
	u64 (*get_supported_physical_layer)(struct txgbe_hw *);
	s32 (*get_mac_addr)(struct txgbe_hw *, u8 *);
	s32 (*get_san_mac_addr)(struct txgbe_hw *, u8 *);
	s32 (*set_san_mac_addr)(struct txgbe_hw *, u8 *);
	s32 (*get_device_caps)(struct txgbe_hw *, u16 *);
	s32 (*get_wwn_prefix)(struct txgbe_hw *, u16 *, u16 *);
	s32 (*get_fcoe_boot_status)(struct txgbe_hw *, u16 *);
	s32 (*read_analog_reg8)(struct txgbe_hw*, u32, u8*);
	s32 (*write_analog_reg8)(struct txgbe_hw*, u32, u8);
	s32 (*setup_sfp)(struct txgbe_hw *);
	s32 (*enable_rx_dma)(struct txgbe_hw *, u32);
	s32 (*disable_sec_rx_path)(struct txgbe_hw *);
	s32 (*enable_sec_rx_path)(struct txgbe_hw *);
	s32 (*disable_sec_tx_path)(struct txgbe_hw *);
	s32 (*enable_sec_tx_path)(struct txgbe_hw *);
	s32 (*acquire_swfw_sync)(struct txgbe_hw *, u32);
	void (*release_swfw_sync)(struct txgbe_hw *, u32);
	void (*init_swfw_sync)(struct txgbe_hw *);
	u64 (*autoc_read)(struct txgbe_hw *);
	void (*autoc_write)(struct txgbe_hw *, u64);
	s32 (*prot_autoc_read)(struct txgbe_hw *, bool *, u64 *);
	s32 (*prot_autoc_write)(struct txgbe_hw *, bool, u64);
	s32 (*negotiate_api_version)(struct txgbe_hw *hw, int api);
	void (*init_mac_link_ops)(struct txgbe_hw *hw);

	/* Link */
	void (*disable_tx_laser)(struct txgbe_hw *);
	void (*enable_tx_laser)(struct txgbe_hw *);
	void (*flap_tx_laser)(struct txgbe_hw *);
	s32 (*setup_link)(struct txgbe_hw *, u32, bool);
	s32 (*setup_mac_link)(struct txgbe_hw *, u32, bool);
	s32 (*check_link)(struct txgbe_hw *, u32 *, bool *, bool);
	s32 (*get_link_capabilities)(struct txgbe_hw *, u32 *,
				     bool *);
	void (*set_rate_select_speed)(struct txgbe_hw *, u32);

	/* Packet Buffer manipulation */
	void (*setup_pba)(struct txgbe_hw *, int, u32, int);

	/* LED */
	s32 (*led_on)(struct txgbe_hw *, u32);
	s32 (*led_off)(struct txgbe_hw *, u32);
	s32 (*blink_led_start)(struct txgbe_hw *, u32);
	s32 (*blink_led_stop)(struct txgbe_hw *, u32);
	s32 (*init_led_link_act)(struct txgbe_hw *);

	/* RAR, Multicast, VLAN */
	s32 (*set_rar)(struct txgbe_hw *, u32, u8 *, u32, u32);
	s32 (*set_uc_addr)(struct txgbe_hw *, u32, u8 *);
	s32 (*clear_rar)(struct txgbe_hw *, u32);
	s32 (*insert_mac_addr)(struct txgbe_hw *, u8 *, u32);
	s32 (*set_vmdq)(struct txgbe_hw *, u32, u32);
	s32 (*set_vmdq_san_mac)(struct txgbe_hw *, u32);
	s32 (*clear_vmdq)(struct txgbe_hw *, u32, u32);
	s32 (*init_rx_addrs)(struct txgbe_hw *);
	s32 (*update_uc_addr_list)(struct txgbe_hw *, u8 *, u32,
				   txgbe_mc_addr_itr);
	s32 (*update_mc_addr_list)(struct txgbe_hw *, u8 *, u32,
				   txgbe_mc_addr_itr, bool clear);
	s32 (*enable_mc)(struct txgbe_hw *);
	s32 (*disable_mc)(struct txgbe_hw *);
	s32 (*clear_vfta)(struct txgbe_hw *);
	s32 (*set_vfta)(struct txgbe_hw *, u32, u32, bool, bool);
	s32 (*set_vlvf)(struct txgbe_hw *, u32, u32, bool, u32 *, u32,
			bool);
	s32 (*init_uta_tables)(struct txgbe_hw *);
	void (*set_mac_anti_spoofing)(struct txgbe_hw *, bool, int);
	void (*set_vlan_anti_spoofing)(struct txgbe_hw *, bool, int);
	s32 (*update_xcast_mode)(struct txgbe_hw *, int);
	s32 (*set_rlpml)(struct txgbe_hw *, u16);

	/* Flow Control */
	s32 (*fc_enable)(struct txgbe_hw *);
	s32 (*setup_fc)(struct txgbe_hw *);
	void (*fc_autoneg)(struct txgbe_hw *);

	/* Manageability interface */
	s32 (*set_fw_drv_ver)(struct txgbe_hw *, u8, u8, u8, u8, u16,
			      const char *);
	s32 (*get_thermal_sensor_data)(struct txgbe_hw *);
	s32 (*init_thermal_sensor_thresh)(struct txgbe_hw *hw);
	void (*get_rtrup2tc)(struct txgbe_hw *hw, u8 *map);
	void (*disable_rx)(struct txgbe_hw *hw);
	void (*enable_rx)(struct txgbe_hw *hw);
	void (*set_source_address_pruning)(struct txgbe_hw *, bool,
					   unsigned int);
	void (*set_ethertype_anti_spoofing)(struct txgbe_hw *, bool, int);
	s32 (*dmac_update_tcs)(struct txgbe_hw *hw);
	s32 (*dmac_config_tcs)(struct txgbe_hw *hw);
	s32 (*dmac_config)(struct txgbe_hw *hw);
	s32 (*setup_eee)(struct txgbe_hw *hw, bool enable_eee);
	s32 (*read_iosf_sb_reg)(struct txgbe_hw *, u32, u32, u32 *);
	s32 (*write_iosf_sb_reg)(struct txgbe_hw *, u32, u32, u32);
	void (*disable_mdd)(struct txgbe_hw *hw);
	void (*enable_mdd)(struct txgbe_hw *hw);
	void (*mdd_event)(struct txgbe_hw *hw, u32 *vf_bitmap);
	void (*restore_mdd_vf)(struct txgbe_hw *hw, u32 vf);
	bool (*fw_recovery_mode)(struct txgbe_hw *hw);

	s32 (*kr_handle)(struct txgbe_hw *hw);
	void (*bp_down_event)(struct txgbe_hw *hw);

	enum txgbe_mac_type type;
	u8 addr[ETH_ADDR_LEN];
	u8 perm_addr[ETH_ADDR_LEN];
	u8 san_addr[ETH_ADDR_LEN];
	/* prefix for World Wide Node Name (WWNN) */
	u16 wwnn_prefix;
	/* prefix for World Wide Port Name (WWPN) */
	u16 wwpn_prefix;
#define TXGBE_MAX_MTA			128
	u32 mta_shadow[TXGBE_MAX_MTA];
	s32 mc_filter_type;
	u32 mcft_size;
	u32 vft_size;
	u32 num_rar_entries;
	u32 rar_highwater;
	u32 rx_pb_size;
	u32 max_tx_queues;
	u32 max_rx_queues;
	u64 autoc;
	u64 orig_autoc;  /* cached value of AUTOC */
	u8  san_mac_rar_index;
	bool get_link_status;
	//u16 max_msix_vectors;
	//bool arc_subsystem_valid;
	bool orig_link_settings_stored;
	bool autotry_restart;
	u8 flags;
	struct txgbe_thermal_sensor_data  thermal_sensor_data;
	bool thermal_sensor_enabled;
	struct txgbe_dmac_config dmac_cfg;
	bool set_lben;
	u32  max_link_up_time;
};

struct txgbe_phy_info {
	u32 (*get_media_type)(struct txgbe_hw *);
	s32 (*identify)(struct txgbe_hw *);
	s32 (*identify_sfp)(struct txgbe_hw *);
	s32 (*init)(struct txgbe_hw *);
	s32 (*reset)(struct txgbe_hw *);
	s32 (*read_reg)(struct txgbe_hw *, u32, u32, u16 *);
	s32 (*write_reg)(struct txgbe_hw *, u32, u32, u16);
	s32 (*read_reg_mdi)(struct txgbe_hw *, u32, u32, u16 *);
	s32 (*write_reg_mdi)(struct txgbe_hw *, u32, u32, u16);
	s32 (*setup_link)(struct txgbe_hw *);
	s32 (*setup_internal_link)(struct txgbe_hw *);
	s32 (*setup_link_speed)(struct txgbe_hw *, u32, bool);
	s32 (*check_link)(struct txgbe_hw *, u32 *, bool *);
	s32 (*get_firmware_version)(struct txgbe_hw *, u32 *);
	s32 (*read_i2c_byte)(struct txgbe_hw *, u8, u8, u8 *);
	s32 (*write_i2c_byte)(struct txgbe_hw *, u8, u8, u8);
	s32 (*read_i2c_sff8472)(struct txgbe_hw *, u8, u8 *);
	s32 (*read_i2c_sff8636)(struct txgbe_hw *, u8, u8, u8 *);
	s32 (*read_i2c_eeprom)(struct txgbe_hw *, u8 , u8 *);
	s32 (*write_i2c_eeprom)(struct txgbe_hw *, u8, u8);
	void (*i2c_bus_clear)(struct txgbe_hw *);
	s32 (*check_overtemp)(struct txgbe_hw *);
	s32 (*set_phy_power)(struct txgbe_hw *, bool on);
	s32 (*enter_lplu)(struct txgbe_hw *);
	s32 (*handle_lasi)(struct txgbe_hw *hw);
	s32 (*read_i2c_byte_unlocked)(struct txgbe_hw *, u8 offset, u8 addr,
				      u8 *value);
	s32 (*write_i2c_byte_unlocked)(struct txgbe_hw *, u8 offset, u8 addr,
				       u8 value);

	enum txgbe_phy_type type;
	u32 addr;
	u32 id;
	enum txgbe_sfp_type sfp_type;
	u32 fiber_suppport_speed;
	bool sfp_setup_needed;
	u32 revision;
	u32 media_type;
	u32 phy_semaphore_mask;
	bool reset_disable;
	u32 autoneg_advertised;
	u32 speeds_supported;
	u32 eee_speeds_supported;
	u32 eee_speeds_advertised;
	enum txgbe_smart_speed smart_speed;
	bool smart_speed_active;
	bool multispeed_fiber;
	bool reset_if_overtemp;
	bool qsfp_shared_i2c_bus;
	u32 nw_mng_if_sel;

	u32 link_mode; //fixme

	/* Some features need tri-state capability */
	u8 ffe_set;
	u8 ffe_main;
	u8 ffe_pre;
	u8 ffe_post;
	u16 fec_mode;
};

#ifdef RTE_TXGBE_FFE_SET
#define TXGBE_FFE_SET	RTE_TXGBE_FFE_SET
#else
#define TXGBE_FFE_SET	TXGBE_BP_M_NULL
#endif
#ifdef RTE_TXGBE_FFE_MAIN
#define TXGBE_FFE_MAIN	RTE_TXGBE_FFE_MAIN
#endif
#ifdef RTE_TXGBE_FFE_PRE
#define TXGBE_FFE_PRE	RTE_TXGBE_FFE_PRE
#endif
#ifdef RTE_TXGBE_FFE_POST
#define TXGBE_FFE_POST	RTE_TXGBE_FFE_POST
#endif
#ifdef RTE_TXGBE_BP_AUTO
#define TXGBE_BP_AN	1
#else
#define TXGBE_BP_AN	0
#endif
#ifdef RTE_TXGBE_KR_FEC
#define TXGBE_KR_FEC 1
#else
#define TXGBE_KR_FEC 0
#endif
#ifdef RTE_TXGBE_KR_RESTART_T_MODE
#define KR_RESTART_T_MODE	RTE_TXGBE_KR_RESTART_T_MODE
#else
#define KR_RESTART_T_MODE 0
#endif
#ifdef RTE_TXGBE_AN73_TRAINNING_MODE
#define AN73_TRAINNING_MODE	RTE_TXGBE_AN73_TRAINNING_MODE
#else
#define AN73_TRAINNING_MODE 1	//0 : kd5886 1: centc 2: wx2wx
#endif
#ifdef RTE_TXGBE_KR_POLL
#define TXGBE_KR_POLL	1
#else
#define TXGBE_KR_POLL	0
#endif
#ifdef RTE_TXGBE_KR_PRESENT
#define TXGBE_KR_PRESENT	1
#else
#define TXGBE_KR_PRESENT	0
#endif
#ifdef RTE_TXGBE_KX_TXRX_PIN
#define TXGBE_KX_TXRX_PIN	1
#else
#define TXGBE_KX_TXRX_PIN	0
#endif
#ifdef RTE_TXGBE_KX_SGMII
#define TXGBE_KX_SGMII	1
#else
#define TXGBE_KX_SGMII	0
#endif
#ifndef TXGBE_DAC_AUTO
#define TXGBE_DAC_AUTO 1
#endif

#define TXGBE_DEVARG_BP_AUTO		"auto_neg"
#define TXGBE_DEVARG_KR_POLL		"poll"
#define TXGBE_DEVARG_KR_PRESENT		"present"
#define TXGBE_DEVARG_KR_FEC		"fec"
#define TXGBE_DEVARG_KX_SGMII		"sgmii"
#define TXGBE_DEVARG_KR_RESTART_T_MODE	"restart_t_mode"
#define TXGBE_DEVARG_AN73_TRAINING_MODE	"an73_training_mode"
#define TXGBE_DEVARG_FFE_SET		"ffe_set"
#define TXGBE_DEVARG_FFE_MAIN		"ffe_main"
#define TXGBE_DEVARG_FFE_PRE		"ffe_pre"
#define TXGBE_DEVARG_FFE_POST		"ffe_post"
#define TXGBE_DEVARG_MSIX_ENABLE	"msix_enable"
#define TXGBE_DEVARG_TX_HEAD_WB		"tx_headwb"
#define TXGBE_DEVARG_TX_HEAD_WB_SIZE	"tx_headwb_size"
#define TXGBE_DEVARG_RX_DESC_MERGE	"rx_desc_merge"
#define TXGBE_DEVARG_DMA_RESET		"dma_reset"
#define TXGBE_DEVARG_RX_WAIT		"rx_wait"

#ifndef TXGBE_SWFW_MBOX_AML
#define TXGBE_SWFW_MBOX_AML
#endif

struct phytxeq {
	u32 main; //TX EQ main (bit[5:0])
	u32 pre1; //TX EQ pre1 (bit[5:0])
	u32 pre2; //TX EQ pre2 (bit[5:0])
	u32 post; //TX EQ post (bit[5:0])
};

struct txgbe_mbx_stats {
	u32 msgs_tx;
	u32 msgs_rx;

	u32 acks;
	u32 reqs;
	u32 rsts;
};

struct txgbe_mbx_info {
	void (*init_params)(struct txgbe_hw *hw);
	s32  (*read)(struct txgbe_hw *, u32 *, u16,  u16);
	s32  (*write)(struct txgbe_hw *, u32 *, u16, u16);
	s32  (*read_posted)(struct txgbe_hw *, u32 *, u16,  u16);
	s32  (*write_posted)(struct txgbe_hw *, u32 *, u16, u16);
	s32  (*check_for_msg)(struct txgbe_hw *, u16);
	s32  (*check_for_ack)(struct txgbe_hw *, u16);
	s32  (*check_for_rst)(struct txgbe_hw *, u16);

	struct txgbe_mbx_stats stats;
	u32 timeout;
	u32 usec_delay;
	u32 v2p_mailbox;
	u16 size;
};

enum txgbe_isb_idx {
	TXGBE_ISB_HEADER,
	TXGBE_ISB_MISC,
	TXGBE_ISB_VEC0,
	TXGBE_ISB_VEC1,
	TXGBE_ISB_MAX
};

#define TXGBE_PHY_FEC_RS	MS(0, 0x1)
#define TXGBE_PHY_FEC_BASER	MS(1, 0x1)
#define TXGBE_PHY_FEC_OFF	MS(2, 0x1)
#define TXGBE_PHY_FEC_AUTO 	(TXGBE_PHY_FEC_OFF | TXGBE_PHY_FEC_BASER |\
				 TXGBE_PHY_FEC_RS)

struct txgbe_devargs {
	u16 auto_neg;
	u16 poll;
	u16 present;
	u16 fec;
	u16 sgmii;
	u16 restart_t_mode;
	u16 an73_training_mode;
	u16 msix_enable;
	u16 tx_headwb;
	u16 tx_headwb_size;
	u16 rx_desc_merge;
	u16 dma_reset;
	u16 rx_wait;
};

struct txgbe_hw {
	void IOMEM *hw_addr;
	void *back;
	struct txgbe_mac_info mac;
	struct txgbe_addr_filter_info addr_ctrl;
	struct txgbe_fc_info fc;
	struct txgbe_phy_info phy;
	struct txgbe_link_info link;
	struct txgbe_rom_info rom;
	struct txgbe_flash_info flash;
	struct txgbe_bus_info bus;
	struct txgbe_mbx_info mbx;
	//const u32 *mvals;
	u16 device_id;
	u16 vendor_id;
	u16 subsystem_device_id;
	u16 subsystem_vendor_id;
	u8 port_id;
	u8 revision_id;
	bool adapter_stopped;
	int api_version;
	bool force_full_reset;
	bool allow_unsupported_sfp;
	bool wol_enabled;
	bool lldp_enabled;
	bool need_crosstalk_fix;
	bool dev_start;
	bool autoneg;  /* autoneg flag set by dpdk application */
	bool an73_training;
	struct txgbe_devargs devarg;

	u32 b4_buf[16];
	uint64_t isb_dma;
	void IOMEM *isb_mem;
	u16 nb_rx_queues;
	u16 nb_tx_queues;
#if 1 /* test by wangjian <begin> */
	u32 fw_version;
	u32 mode;
	enum txgbe_link_status {
		TXGBE_LINK_STATUS_NONE = 0,
		TXGBE_LINK_STATUS_KX,
		TXGBE_LINK_STATUS_KX4
	} link_status;
	enum txgbe_reset_type {
		TXGBE_LAN_RESET = 0,
		TXGBE_SW_RESET,
		TXGBE_GLOBAL_RESET
	} reset_type;

	/* For vector PMD */
	bool rx_bulk_alloc_allowed;
	bool rx_vec_allowed;
	bool tx_vec_allowed;
#endif /* test by wangjian <end> */
	u32 q_rx_regs[128*4];
	u32 q_tx_regs[128*4];
	bool offset_loaded;
	bool rx_loaded;
	struct {
		u64 rx_qp_packets;
		u64 tx_qp_packets;
		u64 rx_qp_bytes;
		u64 tx_qp_bytes;
		u64 rx_qp_mc_packets;
	} qp_last[TXGBE_MAX_QP];

	rte_spinlock_t phy_lock;
	/*amlite: new SW-FW mbox */
	u8 swfw_index;
	rte_atomic32_t swfw_busy;
	bool link_valid;
	bool reconfig_rx;
	/* workaround for temperature alarm */
	bool overheat;
	u32 fec_mode;
	u32 cur_fec_link;
	int temperature;
	u32 bp_link_mode;
	bool dac_sfp;         /* force dac sfp to kr mode */
	bool bypass_ctle; /* DAC cable length */
	u32 curbp_link_mode;
	bool an_done;
	u32 fsm;
	u64 bp_event_interval;
};

typedef enum {
	ABILITY_1000BASE_KX,
	ABILITY_10GBASE_KX4,
	ABILITY_10GBASE_KR,
	ABILITY_40GBASE_KR4,
	ABILITY_40GBASE_CR4,
	ABILITY_100GBASE_CR10,
	ABILITY_100GBASE_KP4,
	ABILITY_100GBASE_KR4,
	ABILITY_100GBASE_CR4,
	ABILITY_25GBASE_KRCR_S,
	ABILITY_25GBASE_KRCR,
	ABILITY_MAX,
} ability_filed_encding;

struct txgbe_backplane_ability {
	u32 next_page;	  //Next Page (bit0)
	u32 link_ability; //Link Ability (bit[7:0])
	u32 fec_ability;  //FEC Request (bit1), FEC Enable  (bit0)
	u32 current_link_mode; //current link mode for local device
};

#include "txgbe_regs.h"
#include "txgbe_dummy.h"

#endif /* _TXGBE_TYPE_H_ */
